/* 各県の毎日の降水量のデータ（2019年1月1日〜12月31日）を読み込んで、その変化を「雨の表現」を使って表示する */
let prefCSV;  //各県の名前、県庁所在地、xy座標が入ったcsvデータを読み込み、保存しておくための変数
let prefs;    //Prefectureクラスのインスタンスを入れる配列
let dataCSV;  //各県の日毎の降水量が入ったcsvデータ（気象庁HPからダウンロード）を読み込み、保存しておくための変数
let curYear = 2019;   //表示上で「今○年」を表すための変数
let curMonth = 1;   //表示上で「今○月」を表すための変数
let curDay = 1;   //表示上で「今○日」を表すための変数
let time = 0;   //経過フレーム数をカウントするための変数（これが一定の数に増えた時に、curDayを1増やす）

function preload(){
  prefCSV = loadTable("prefectures.csv", "csv", "header");    //各県の県庁所在地データを読み込む
  dataCSV = loadTable("kosuiryo_2019_zenkoku.csv", "csv");    //各県の毎日の降水量を読み込む（1行目をヘッダーとして扱わない場合は引数を2つにする）
}

function setup(){
  createCanvas(windowWidth, windowHeight);  //ウィンドウサイズの指定
  let kenNum = prefCSV.getRowCount();   //各県のCSVデータの行数を取得する
  prefs = new Array(kenNum);    //prefsを初期化する。要素の数はデータの行数と同じにする
  for(let i = 0; i < prefs.length; i++){
    let row = prefCSV.getRow(i);    //1行分のデータを取得
    prefs[i] = new Prefecture();
    prefs[i].x = row.getNum("x");   //「x」の値を取得
    prefs[i].y = row.getNum("y");   //「y」の値を取得
    prefs[i].name = row.getString("都道府県名");   //「都道府県名」の値を取得
    prefs[i].kenchoName = row.getString("県庁所在地");   //「県庁所在地」の値を取得
  }

  let rowNum = dataCSV.getRowCount();   //降水量のCSVデータの行数を取得
  let columnNum = dataCSV.getColumnCount();   //CSVデータの列数を取得
  let headerRow = dataCSV.getRow(0);    //ヘッダー行のデータ（観測地点名）を取っておく
  //以下、降水量データを1つずつ順に読み込んで、該当する県（prefs）の「dataList」変数に入れていく
  for(let i = 0; i < rowNum; i++){
    if(i > 2){    //3行目以降だったら（0〜2行目は降水量データではないので何もしない）
      let row = dataCSV.getRow(i);  //1行分のデータを取得
      let dateSt = "";
      let lastKenchoSt = "";
      for(let j = 0; j < columnNum; j++){
        //0列目は日付なので、それをdateStとして取っておく
        if(j == 0){
          dateSt = row.getString(j);
        }
        kenchoSt = headerRow.getString(j);  //ヘッダー行の同列の値（観測地点名）を取る
        if(kenchoSt != lastKenchoSt){   //1ループ前の列の観測地点名と今の列の観測地点名が違ったら、それが降水量のデータなので、読み込む
          let kosuiData = new KosuiData();      //新しい降水データの入れ物を作る
          let dateTokens = dateSt.split("/");   //dateSt（日付の文字列）を「/」で分割して、配列にする→「年」「月」「日」が順に入った配列になる
          kosuiData.year = int(dateTokens[0]);   //年
          kosuiData.month = int(dateTokens[1]);  //月
          kosuiData.day = int(dateTokens[2]);    //日
          kosuiData.kosuiryo = row.getNum(j);    //降水量データを取得する
          //同じ県庁所在地名の県を探して、そのdataListに新しい降水データを追加する
          for(let k = 0; k < prefs.length; k++){
            if(kenchoSt == prefs[k].kenchoName){
              let pref = prefs[k];
              kosuiData.pref = pref;
              pref.dataList.push(kosuiData);
              break;  //for文を抜ける
            }
          }
        }
        lastKenchoSt = kenchoSt;  //「1ループ前の観測地点名」を取っておく
      }
    }
  }
}

function draw(){
  background(0);
  //「現在の日付」の降水データを描画する
  for(let i = 0; i < prefs.length; i++){
    prefs[i].drawDataOfTheDay(curYear, curMonth, curDay);
  }

  //一定のフレーム数が経過したら「現在の日付」を1日進める
  time += 1;  //時間をカウントする
  if(time > 60){
    curDay += 1;
    time = 0;
    let daysOfMonth = 30; //その月の日数
    if(curMonth == 1 || curMonth == 3 || curMonth == 5 || curMonth == 7 || curMonth == 8 || curMonth == 10 || curMonth == 12){
      daysOfMonth = 31;
    }else if(curMonth == 2){
      daysOfMonth = 28;
    }
    if(curDay > daysOfMonth){
      curDay = 1;
      curMonth += 1;
    }
  }
  fill(255);
  noStroke();
  textSize(24);
  text(curYear + "." + curMonth + "." + curDay, 50, 50);
}

//各県のデータを入れるためのクラス
class Prefecture {
  constructor(){
    this.x = 0;       //県庁所在地のx座標
    this.y = 0;       //県庁所在地のy座標
    this.name = "";   //都道府県名
    this.kenchoName = "";   //県庁所在地の名前
    this.dataList = new Array(0); //KosuiDataのリスト
    this.pts = new Array(200);      //雨の点のxy座標を入れておく
    this.addX = new Array(this.pts.length);   //雨に少し揺らぎを与えるため、毎フレームaddXを加える
    this.drawingPtNum = 0;    //この番号までの雨を降らせる
    this.iDrawingPtNum = 0;    //drawingPtNumの目標値（ゆっくり変化させるため）
    this.hankei = 30;   //雨のエリアの半径
    //雨の点の座標をランダムで決める
    for(let i = 0; i < this.pts.length; i++){
      let r = random(this.hankei);   //半径をランダムで決める
      let angle = random(2 * PI);   //角度をランダムで決める
      this.pts[i] = createVector(r * cos(angle), r * sin(angle));   //半径と角度からx,yを求める
      this.addX[i] = random(-0.2, 0.2);   //雨に揺らぎを与えるためのaddXをランダムで決める
    }
  }
  //引数の日付のデータを描画するメソッド
  drawDataOfTheDay(year, month, day){
    //dataListから該当する日付のデータを探し、描画すべき雨の数を割り出す
    for(let i = 0; i < this.dataList.length; i++){
      let data = this.dataList[i];
      if(data.year == year && data.month == month && data.day == day){
        let kosuiryo = data.kosuiryo;
        this.iDrawingPtNum = kosuiryo * 2;
      }
    }
    //描画する雨の数を、目標値にゆっくり変化させる
    this.drawingPtNum += (this.iDrawingPtNum - this.drawingPtNum) / 20.0;
    //雨を描画する
    for(let i = 0; i < this.pts.length; i++){
      if(i < this.drawingPtNum){
        let pt = this.pts[i];
        noFill();
        stroke(0, 100, 255);
        point(this.x + pt.x, this.y + pt.y);  //雨を点として描く
        pt.x += this.addX[i];   //雨の点を左右に揺らがせる
        pt.y += 2;    //雨の点を下に動かす
        let r = dist(0, 0, pt.x, pt.y);
        //任意の半径から外に出てしまったら、円の上弦の上に雨を移動させる
        if(r > this.hankei){
          let angle = random(PI, 2*PI);
          pt.x = this.hankei * cos(angle);
          pt.y = this.hankei * sin(angle);
        }
      }
    }
  }
}
//降水量のデータを入れるクラス
class KosuiData {
  constructor(){
    this.year = 1970;   //降水データの年号
    this.month = 1;   //降水データの月
    this.day = 1;   //降水データの日
    this.kosuiryo = 0;  //降水量
    this.pref = null;   //そのデータの県
  }
}
