/* 量の変化を時間経過に沿って見せるダイアグラム  */
let cx = 0.0;      //円の中心座標のx
let cy = 0.0;      //同、y
let r = 300.0;     //円の半径
let totalArea = 0; //円の総面積

let pts;      //各パーティクルの座標（円の中心を基準とする）
let pArea = 0; //一人当たり面積（平均面積）
let pR = 0;    //一人当たり半径
let pAlpha = 0.0;         //各パーティクルの個人空間を示す円のアルファ値

let densityData = [30, 80, 150, 300, 500];  //密度のデータ
let dataEpochs = [0, 300, 600, 1200, 1800];    //密度のデータ年代
let curEpoch = 0;  //現在の年代区切り（データ年代の何番目か）
let time = 0;      //現在の年代

let isPressed = false;  //マウスを押しているか

function setup(){
  createCanvas(windowWidth, windowHeight);  //ウィンドウサイズの指定
  colorMode(HSB);
  cx = width/2;      //円の中心を画面の中心に
  cy = height/2;
  pts = new Array(int(densityData[0]));    //最初の密度データのぶんだけパーティクルを作る
  //パーティクルをランダムで配置する==================
  for (let i = 0; i < pts.length; i++) {
    let randomR = random(r);
    let randomAngle = random(PI*2);
    pts[i] = createVector(randomR * cos(randomAngle), randomR * sin(randomAngle));
  }
  totalArea = r * r * PI;        //円の総面積の計算
  pArea = totalArea / pts.length; //平均面積の算出
  pR = sqrt(pArea / PI);         //一人当たりの個人空間の半径。
}

function draw(){
  background(0);

  //左下のテキスト表示
  fill(255);
  noStroke();
  textAlign(LEFT);
  text("time: " + time + "\nParticles: " + densityData[curEpoch] + "\nPersonal Area: " + pArea, 10, height-50);

  //時代が変わるとその時代の密度データに合わせて点の数を変える
  if (curEpoch < dataEpochs.length -1) {
    if (dataEpochs[curEpoch+1] < time) {
      onDataEpochChanged();
    }
  }

  //マウスを押している間は不透明度を255に近づける。それ以外は0に近づける
  if (isPressed == true) {
    pAlpha += (255.0 - pAlpha) / 20.0;
  } else {
    pAlpha += (0.0 - pAlpha) / 20.0;
  }
  //各パーティクルの描画
  for (let i = 0; i < pts.length; i++) {
    let pt = pts[i];
    //自分の位置に点を描く
    noStroke();
    fill(100, 100, 200, 100);
    circle(cx + pt.x, cy + pt.y, 4);
    //パーソナルエリアの大きさで縁を描く
    stroke(30, 100, 200, pAlpha);
    noFill();
    circle(cx + pt.x, cy + pt.y, pR*2);
    //ランダム方向に移動
    pt.x += random(-1, 1);
    pt.y += random(-1, 1);
    //中心からの距離が半径より広くなってしまったら、円の反対側に移動する
    let d = sqrt(pow(pt.x, 2) + pow(pt.y, 2));  //中心からの距離を測る
    if (d > r) {
      let angle = atan2(pt.y, pt.x);
      angle += PI;
      pt.x = r * cos(angle);
      pt.y = r * sin(angle);
    }
    //他の点との距離を測って、平均距離より狭くなってしまったら反発する
    for (let j = 0; j < pts.length; j++) {
      if (i != j) {
        let ppt = pts[j];
        let d2 = dist(pt.x, pt.y, ppt.x, ppt.y);
        if (d2 < pR*2) {
          let angle2 = atan2(ppt.y - pt.y, ppt.x - pt.x);
          angle2 += PI;
          pt.x += cos(angle2);
          pt.y += sin(angle2);
        }
      }
    }
  }

  //円
  stroke(180, 255, 255);
  noFill();
  circle(cx, cy, r*2);

  //下の目盛り
  let lineW = 200;
  let lineX = width/2 - lineW/2;
  let lineY = height-50;
  let lineH = 5;
  let totalEpoch = dataEpochs[dataEpochs.length-1] - dataEpochs[0];  //最初と最後の時代の年数の差を算出
  stroke(20, 255, 255);
  noFill();
  line(lineX, lineY, lineX + lineW, lineY);  //ベースの線
  textAlign(CENTER);
  //各目盛の描画
  for (let i = 0; i < dataEpochs.length; i++) {
    stroke(20, 255, 255);
    noFill();
    let memoriX = lineX + lineW/totalEpoch * (dataEpochs[i] - dataEpochs[0]);  //目盛のx座標計算
    line(memoriX, lineY - lineH, memoriX, lineY);
    noStroke();
    if (i == curEpoch) {
      fill(255);
    } else {
      fill(0, 0, 150);
    }
    text(dataEpochs[i], memoriX, lineY - 15);  //年号の表示
  }

  //現在の年数を示す三角形
  let  nowX = lineX + lineW/totalEpoch * (time - dataEpochs[0]);  //x座標の計算
  if(nowX > lineX + lineW){
    nowX = lineX + lineW;
  }
  let nowY = lineY;
  let triL = 10;
  noStroke();
  fill(0, 255, 255);
  triangle(nowX, nowY, nowX + triL * cos(PI/3), nowY + triL * sin(PI/3), nowX - triL * cos(PI/3), nowY + triL * sin(PI/3));

  time += 1;
}

function onDataEpochChanged(){
  let newPts = new Array(int(densityData[curEpoch+1]));
  for (let i = 0; i < newPts.length; i++) {
    if (i < pts.length) {   //これまでのptsの長さの範囲まではそれをコピーする
      newPts[i] = pts[i];
    } else {      //これまでのptsの長さを超えるぶんに関してはランダムで位置を決める
      let randomR = random(r);
      let randomAngle = random(PI*2);
      newPts[i] = createVector(randomR * cos(randomAngle), randomR * sin(randomAngle));
    }
  }
  pts = newPts;
  pArea = totalArea / pts.length;
  pR = sqrt(pArea / PI);
  curEpoch += 1;
}

function mousePressed(){
  isPressed = true;
}

function mouseReleased(){
  isPressed = false;
}
